/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Text;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using System.Web.SessionState;
using Borland.Eco.Handles;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.UmlRt;
using Borland.Eco.Web;
using System.Globalization;

namespace Borland.Eco.Web
{

	public enum EcoPropertyType
	{
		Undefined = 0,
		Attribute = 1,
		SingleLink = 2,
		MultiLink = 3
	}

	/// <summary>
	/// Summary description for WebUtils.
	/// </summary>
	public sealed class WebUtils
	{
		private WebUtils() { }
		///<exception cref="ArgumentNullException">Thrown if <paramref name="property"/> is null.</exception>
		public static EcoPropertyType GetPropertyType(IProperty property)
		{
			if (property == null)
				throw new ArgumentNullException("property");
			IFeature sf = property.StructuralFeature;

			if ((sf.FeatureType == FeatureType.AssociationEnd) &&
				(((IAssociationEnd)sf).Multiplicity.Upper > 1)  &&
				(((IAssociationEnd)sf).AssociationEndKind != AssociationEndKind.ToLinkObject))
			{
				return EcoPropertyType.MultiLink;
			}
			else if (sf.FeatureType == FeatureType.Attribute)
			{
				return EcoPropertyType.Attribute;
			}
			else if ((sf.FeatureType == FeatureType.AssociationEnd) && (((IAssociationEnd)sf).Multiplicity.Upper == 1))
			{
				return EcoPropertyType.SingleLink;
			}
			return EcoPropertyType.Undefined;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		public static IObject ObjectForId(EcoSpace ecoSpace, string id)
		{
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			IExternalIdService idService = ecoSpace.GetEcoService(typeof(IExternalIdService)) as IExternalIdService;
			return idService.ObjectForId(id);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		public static string IdForObject(EcoSpace ecoSpace, IObject obj)
		{
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			IExternalIdService idService = ecoSpace.GetEcoService(typeof(IExternalIdService)) as IExternalIdService;
			return idService.IdForObject(obj);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		public static string GetObjectDisplayName(EcoSpace ecoSpace, IObject obj)
		{
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			IOclService oclSvc = ecoSpace.GetEcoService(typeof(IOclService)) as IOclService;
			return oclSvc.Evaluate(obj, "self.asString").AsObject.ToString(); // do not localize
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="obj"/> is null.</exception>
		public static string GetObjectClassName(IObject obj)
		{
			if (obj == null)
				throw new ArgumentNullException("obj");
			return obj.UmlClass.Name;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="request"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="obj"/> is null.</exception>
		public static HyperLink GetObjectClassLink(HttpRequest request,  EcoSpace ecoSpace, IObject obj)
		{
			if (request == null)
				throw new ArgumentNullException("request");
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			if (obj == null)
				throw new ArgumentNullException("obj");

			string className = GetObjectClassName(obj);
			
			HyperLink link = ControlBuilder.MakeNewHyperLink("Class" + className + "Link", null); // do not localize
			//link.CssClass = EcoCssClassNames.EcoAutoFormLink;

			link.NavigateUrl = String.Format(CultureInfo.InvariantCulture, "{0}?{1}={2}", request.Url.LocalPath, EcoQueryVariableNames.Class, obj.UmlType.Name); // do not localize
			link.Text = className;

			return link;
		}
		
		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		public static void SaveSpace(EcoSpace ecoSpace)
		{
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			IPersistenceService persistenceSvc = ecoSpace.GetEcoService(typeof(IPersistenceService)) as IPersistenceService;
			IDirtyListService dlSvc = ecoSpace.GetEcoService(typeof(IDirtyListService)) as IDirtyListService;
			persistenceSvc.UpdateDatabaseWithList(dlSvc.AllDirtyObjects());
		}

		public static void AddHeaderRow(Table table, string title, string text, HttpRequest request, HttpSessionState session)
		{
			TableRow tableRow = ControlBuilder.MakeNewTableRow("RowHeader" + title, table); // do not localize
			TableCell cell = ControlBuilder.MakeNewTableCell("CellHeader" + title, tableRow); // do not localize
			cell.CssClass = EcoCssClassNames.EcoAutoFormHeaderCell; // do not localize
			cell.ColumnSpan = 2;

			AddHeaderTableToCell(cell, title, text, request, session);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="region"/> is null.</exception>
		public static TableCell AddHeaderTableToCell(TableCell tableCell, string title, string text, HttpRequest request, HttpSessionState session)
		{
			if (session == null)
				throw new ArgumentNullException("session"); // do not localize
			Table headerTable = ControlBuilder.MakeNewTable("Table" + title, tableCell); // do not localize
			headerTable.CssClass = EcoCssClassNames.EcoAutoFormTable; // do not localize
			TableRow row = ControlBuilder.MakeNewTableRow("RowInnerHeader" + title, headerTable); // do not localize

			TableCell labelCell = ControlBuilder.MakeNewTableCell("CellLabelHeader" + title, row); // do not localize
			TableCell buttonCell = ControlBuilder.MakeNewTableCell("CellButtonHeader" + title, row); // do not localize
			labelCell.CssClass = EcoCssClassNames.EcoAutoFormHeaderLabelCell; // do not localize
			buttonCell.CssClass = EcoCssClassNames.EcoAutoFormHeaderButtonCell; // do not localize

			headerTable.Width = Unit.Percentage(100);

			Label lbl = ControlBuilder.MakeNewLabel(labelCell, text);
			lbl.CssClass = EcoCssClassNames.EcoAutoFormHeaderLabel;
			HyperLink link = ControlBuilder.MakeNewHyperLink("LinkExpand" + title , buttonCell);
			link.CssClass = EcoCssClassNames.EcoAutoFormHeaderButton;

			if (session["Collapse" + title] != null)
			{
				link.Text = "+";
			}
			else
			{
				link.Text = "-";
			}
			link.NavigateUrl = GetUrlForHeaderButton(session["Collapse" + title] == null, title, request);

			return labelCell;
		}

		private static string GetUrlForHeaderButton(bool expanded, string title, HttpRequest request)
		{
			StringBuilder query = new StringBuilder(ReintroduceParameter(EcoQueryVariableNames.RootId, request));
			query.Append(ReintroduceParameter(EcoQueryVariableNames.OwnerId, request));
			query.Append(ReintroduceParameter(EcoQueryVariableNames.OwnerParameter, request));
			query.Append(ReintroduceParameter(EcoQueryVariableNames.Class, request));

			if (expanded)
				query.Append(String.Format(CultureInfo.InvariantCulture, "{0}={1}", EcoQueryVariableNames.Collapse, title)); // do not localize
			else
				query.Append("Expand=" + title); // do not localize

			string qry = String.Empty;

			if (query.Length > 0)
				qry = "?" + query.ToString() ;

			return request.Url.LocalPath + qry;
		}

		private static string ReintroduceParameter(string paramName, HttpRequest request)
		{
			string param = "";
			string val = request.Params[paramName];
			if (val == null) { val = ""; }
			if (val != "")
				param = String.Format(CultureInfo.InvariantCulture, "{0}={1}&", paramName, val);
			return param;
		}

		public static bool IsAbstractOrLinkClass(EcoSpace ecoSpace, string className)
		{
			return IsAbstractClass(ecoSpace, className) || IsLinkClass(ecoSpace, className);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		public static bool IsAbstractClass(EcoSpace ecoSpace, string className)
		{
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			IEcoClassifier ecoClassifier = ecoSpace.TypeSystem.GetClassifierByName(className).EcoClassifier;
			return ecoClassifier.IsAbstract;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="ecoSpace"/> is null.</exception>
		public static bool IsLinkClass(EcoSpace ecoSpace, string className)
		{
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			IClassifier classifier = ecoSpace.TypeSystem.GetClassifierByName(className);
			IClass cls = classifier as IClass;
			return cls.IsAssociationClass;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="session"/> is null.</exception>
		public static bool IsCollapsedSection(HttpSessionState session, string title)
		{
			if (session == null)
				throw new ArgumentNullException("session");
			return (session[EcoQueryVariableNames.Collapse + title] != null);
		}
	}
}
